<?php

namespace App\Libraries;

use ZipArchive;
use CodeIgniter\HTTP\Response;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Config\Services;

class ZipLibrary
{
    protected $zip;
    protected $zipFileName;
    protected $tempFile;

    public function __construct($zipFileName = 'archive.zip')
    {
        $this->zip = new ZipArchive();
        $this->tempFile = tempnam(sys_get_temp_dir(), 'zip'); // Create temporary file
        $this->zipFileName = $zipFileName;

        if ($this->zip->open($this->tempFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            throw new \Exception("Cannot create ZIP file: " . $this->zipFileName);
        }
    }

    // Add a file from a path
    public function addFile($filePath, $fileNameInZip = null)
    {
        if (file_exists($filePath)) {
            $fileNameInZip = $fileNameInZip ?? basename($filePath);
            $this->zip->addFile($filePath, $fileNameInZip);
            return true;
        }
        return false;
    }

    // Add raw data as a file
    public function addData($fileName, $data)
    {
        $this->zip->addFromString($fileName, $data);
    }

    // Close ZIP archive
    public function close()
    {
        $this->zip->close();
        return $this->tempFile;
    }

    // Download ZIP file and delete it after download
    public function download(): ResponseInterface
    {
        $this->close(); // Ensure ZIP is finalized

        $response = Services::response()
            ->setHeader('Content-Type', 'application/zip')
            ->setHeader('Content-Disposition', 'attachment; filename="' . $this->zipFileName . '"')
            ->setHeader('Content-Length', filesize($this->tempFile))
            ->setBody(file_get_contents($this->tempFile));

        unlink($this->tempFile); // Delete temp file after download
        return $response;
    }
}
